<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../includes/helpers.php';

header('Content-Type: application/json');

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(false, 'Method not allowed', null, 405);
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate required fields (support both camelCase and snake_case)
$name = $input['name'] ?? '';
$emailOrPhone = $input['emailOrPhone'] ?? $input['email_or_phone'] ?? $input['email'] ?? $input['phone'] ?? '';
$password = $input['password'] ?? '';
$isEmail = $input['isEmail'] ?? $input['is_email'] ?? (isset($input['email']) ? true : false);

// Validation
if (empty($name)) {
    sendResponse(false, 'Name is required');
}

if (empty($emailOrPhone)) {
    sendResponse(false, ($isEmail ? 'Email' : 'Phone number') . ' is required');
}

if (empty($password)) {
    sendResponse(false, 'Password is required');
}

// Validate name length
if (strlen(trim($name)) < 2) {
    sendResponse(false, 'Name must be at least 2 characters');
}

if (strlen($name) > 100) {
    sendResponse(false, 'Name must not exceed 100 characters');
}

// Validate email or phone
if ($isEmail) {
    if (!validateEmail($emailOrPhone)) {
        sendResponse(false, 'Invalid email address');
    }
} else {
    if (!validatePhone($emailOrPhone)) {
        sendResponse(false, 'Invalid phone number');
    }
}

// Validate password
if (strlen($password) < 6) {
    sendResponse(false, 'Password must be at least 6 characters');
}

if (strlen($password) > 50) {
    sendResponse(false, 'Password must not exceed 50 characters');
}

try {
    $db = getDBConnection();
    
    // Check if user already exists
    if ($isEmail) {
        $stmt = $db->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute([$emailOrPhone]);
    } else {
        $stmt = $db->prepare("SELECT id FROM users WHERE phone = ?");
        $stmt->execute([$emailOrPhone]);
    }
    
    if ($stmt->fetch()) {
        sendResponse(false, ($isEmail ? 'Email' : 'Phone number') . ' already registered');
    }
    
    // Generate OTP
    $otp = generateOTP();
    $otpExpiry = date('Y-m-d H:i:s', strtotime('+' . OTP_EXPIRY_MINUTES . ' minutes'));
    
    // Hash password
    $hashedPassword = hashPassword($password);
    
    // Begin transaction
    $db->beginTransaction();
    
    // Insert user
    $stmt = $db->prepare("
        INSERT INTO users (name, email, phone, password_hash, is_verified, created_at, updated_at)
        VALUES (?, ?, ?, ?, 0, NOW(), NOW())
    ");
    
    $email = $isEmail ? $emailOrPhone : null;
    $phone = $isEmail ? null : $emailOrPhone;
    
    $stmt->execute([$name, $email, $phone, $hashedPassword]);
    $userId = $db->lastInsertId();
    
    // Insert OTP verification record
    $stmt = $db->prepare("
        INSERT INTO otp_verification (user_id, email, phone, otp, purpose, expires_at, created_at)
        VALUES (?, ?, ?, ?, 'registration', ?, NOW())
    ");

    $stmt->execute([$userId, $email, $phone, $otp, $otpExpiry]);
    
    // Create user settings with default values
    $stmt = $db->prepare("
        INSERT INTO user_settings (user_id, language, theme, created_at, updated_at)
        VALUES (?, 'en', 'light', NOW(), NOW())
    ");
    $stmt->execute([$userId]);
    
    // Create user profile
    $stmt = $db->prepare("
        INSERT INTO user_profiles (user_id, created_at, updated_at)
        VALUES (?, NOW(), NOW())
    ");
    $stmt->execute([$userId]);
    
    // Commit transaction
    $db->commit();
    
    // Send OTP via email or SMS
    if ($isEmail) {
        // TODO: Send email with OTP
        // For now, we'll just log it
        error_log("OTP for $emailOrPhone: $otp");
    } else {
        // TODO: Send SMS with OTP
        // For now, we'll just log it
        error_log("OTP for $emailOrPhone: $otp");
    }
    
    // Log activity
    logActivity($userId, 'user_registered', 'User registered successfully');
    
    // Return success response (don't send OTP in production)
    sendResponse(true, 'Registration successful. Please verify your ' . ($isEmail ? 'email' : 'phone number'), [
        'user_id' => $userId,
        'otp' => $otp // Remove this in production
    ]);
    
} catch (PDOException $e) {
    if (isset($db) && $db->inTransaction()) {
        $db->rollBack();
    }
    error_log("Registration error: " . $e->getMessage());
    sendResponse(false, 'Registration failed. Please try again later', null, 500);
}

