<?php
/**
 * Get BioDynamic Calendar Data
 * GET /api/calendar/get-calendar.php
 */

require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/helpers.php';
require_once __DIR__ . '/../../includes/jwt.php';

header('Content-Type: application/json');
requireAuth();

try {
    $db = getDBConnection();
    
    // Get query parameters
    $month = isset($_GET['month']) ? (int)$_GET['month'] : date('n');
    $year = isset($_GET['year']) ? (int)$_GET['year'] : date('Y');
    
    // Validate month and year
    if ($month < 1 || $month > 12) {
        sendResponse(false, 'Invalid month');
        exit;
    }
    
    if ($year < 2000 || $year > 2100) {
        sendResponse(false, 'Invalid year');
        exit;
    }
    
    // Get calendar data for the month
    $stmt = $db->prepare("
        SELECT
            id,
            date,
            lunar_phase,
            zodiac_sign,
            element,
            is_favorable_planting,
            is_favorable_harvesting,
            is_favorable_pruning,
            is_favorable_fertilizing,
            recommendations,
            notes
        FROM biodynamic_calendar
        WHERE MONTH(`date`) = ? AND YEAR(`date`) = ?
        ORDER BY `date` ASC
    ");
    
    $stmt->execute([$month, $year]);
    $calendarData = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Convert boolean fields
    foreach ($calendarData as &$day) {
        $day['is_favorable_planting'] = (bool)$day['is_favorable_planting'];
        $day['is_favorable_harvesting'] = (bool)$day['is_favorable_harvesting'];
        $day['is_favorable_pruning'] = (bool)$day['is_favorable_pruning'];
        $day['is_favorable_fertilizing'] = (bool)$day['is_favorable_fertilizing'];
        
        // Parse recommendations JSON
        if ($day['recommendations']) {
            $day['recommendations'] = json_decode($day['recommendations'], true);
        } else {
            $day['recommendations'] = [];
        }
    }
    
    sendResponse(true, 'Calendar data retrieved successfully', [
        'month' => $month,
        'year' => $year,
        'calendar' => $calendarData
    ]);
    
} catch (PDOException $e) {
    error_log('Database error: ' . $e->getMessage());
    sendResponse(false, 'Failed to retrieve calendar data');
} catch (Exception $e) {
    error_log('Error: ' . $e->getMessage());
    sendResponse(false, 'An error occurred');
}


