<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/jwt.php';
require_once __DIR__ . '/../../includes/helpers.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    sendErrorResponse('Method not allowed', 405);
}

try {
    // Verify JWT token
    $userId = verifyJWT();
    
    // Get query parameters
    if (!isset($_GET['friend_id'])) {
        sendErrorResponse('Friend ID is required', 400);
    }
    
    $friendId = (int)$_GET['friend_id'];
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
    $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
    
    // Validate limit
    if ($limit < 1 || $limit > 100) {
        $limit = 50;
    }
    
    // Get database connection
    $db = getDBConnection();
    
    // Get messages between user and friend
    $stmt = $db->prepare("
        SELECT 
            m.id, m.sender_id, m.receiver_id, m.message, m.image_url, m.is_read, m.created_at,
            u.name as sender_name
        FROM chat_messages m
        INNER JOIN users u ON m.sender_id = u.id
        WHERE (m.sender_id = ? AND m.receiver_id = ?) OR (m.sender_id = ? AND m.receiver_id = ?)
        ORDER BY m.created_at DESC
        LIMIT ? OFFSET ?
    ");
    $stmt->execute([$userId, $friendId, $friendId, $userId, $limit, $offset]);
    $messages = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Convert boolean fields and reverse order (oldest first)
    $messages = array_reverse($messages);
    foreach ($messages as &$message) {
        $message['is_read'] = (bool)$message['is_read'];
    }
    
    // Mark messages as read
    $stmt = $db->prepare("
        UPDATE chat_messages 
        SET is_read = 1 
        WHERE sender_id = ? AND receiver_id = ? AND is_read = 0
    ");
    $stmt->execute([$friendId, $userId]);
    
    // Get total count
    $stmt = $db->prepare("
        SELECT COUNT(*) as total 
        FROM chat_messages 
        WHERE (sender_id = ? AND receiver_id = ?) OR (sender_id = ? AND receiver_id = ?)
    ");
    $stmt->execute([$userId, $friendId, $friendId, $userId]);
    $total = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    $data = [
        'messages' => $messages,
        'total' => (int)$total,
        'limit' => $limit,
        'offset' => $offset,
    ];
    
    sendSuccessResponse('Messages retrieved successfully', $data);
    
} catch (Exception $e) {
    sendErrorResponse($e->getMessage(), 500);
}


