<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/jwt.php';
require_once __DIR__ . '/../../includes/helpers.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    sendErrorResponse('Method not allowed', 405);
}

try {
    // Verify JWT token
    $userId = verifyJWT();
    
    // Get query parameters
    $category = isset($_GET['category']) ? trim($_GET['category']) : null;
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 20;
    $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
    
    // Validate limit
    if ($limit < 1 || $limit > 100) {
        $limit = 20;
    }
    
    // Get database connection
    $db = getDBConnection();
    
    // Build query
    $query = "
        SELECT 
            p.id, p.title, p.content, p.category, p.image_url, p.created_at, p.updated_at,
            u.id as user_id, u.name as user_name, u.email as user_email,
            (SELECT COUNT(*) FROM community_likes WHERE post_id = p.id) as likes_count,
            (SELECT COUNT(*) FROM community_comments WHERE post_id = p.id) as comments_count,
            (SELECT COUNT(*) FROM community_favorites WHERE post_id = p.id AND user_id = ?) as is_favorited,
            (SELECT COUNT(*) FROM community_likes WHERE post_id = p.id AND user_id = ?) as is_liked
        FROM community_posts p
        INNER JOIN users u ON p.user_id = u.id
    ";
    
    $params = [$userId, $userId];
    
    // Add category filter if provided
    if ($category) {
        $query .= " WHERE p.category = ?";
        $params[] = $category;
    }
    
    $query .= " ORDER BY p.created_at DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    
    $stmt = $db->prepare($query);
    $stmt->execute($params);
    $posts = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Convert boolean fields
    foreach ($posts as &$post) {
        $post['is_favorited'] = (bool)$post['is_favorited'];
        $post['is_liked'] = (bool)$post['is_liked'];
        $post['likes_count'] = (int)$post['likes_count'];
        $post['comments_count'] = (int)$post['comments_count'];
    }
    
    // Get total count
    $countQuery = "SELECT COUNT(*) as total FROM community_posts";
    if ($category) {
        $countQuery .= " WHERE category = ?";
        $countStmt = $db->prepare($countQuery);
        $countStmt->execute([$category]);
    } else {
        $countStmt = $db->query($countQuery);
    }
    $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    $data = [
        'posts' => $posts,
        'total' => (int)$total,
        'limit' => $limit,
        'offset' => $offset,
    ];
    
    sendSuccessResponse('Posts retrieved successfully', $data);
    
} catch (Exception $e) {
    sendErrorResponse($e->getMessage(), 500);
}


