<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/jwt.php';
require_once __DIR__ . '/../../includes/helpers.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendErrorResponse('Method not allowed', 405);
}

try {
    // Verify JWT token
    $userId = verifyJWT();
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Validate required fields
    if (!isset($input['friend_id'])) {
        sendErrorResponse('Friend ID is required', 400);
    }
    
    $friendId = (int)$input['friend_id'];
    
    // Cannot send request to self
    if ($friendId === $userId) {
        sendErrorResponse('Cannot send friend request to yourself', 400);
    }
    
    // Get database connection
    $db = getDBConnection();
    
    // Check if friend exists
    $stmt = $db->prepare("SELECT id, name FROM users WHERE id = ?");
    $stmt->execute([$friendId]);
    $friend = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$friend) {
        sendErrorResponse('User not found', 404);
    }
    
    // Check if already friends or request exists
    $stmt = $db->prepare("
        SELECT id, status 
        FROM friends 
        WHERE (user_id = ? AND friend_id = ?) OR (user_id = ? AND friend_id = ?)
    ");
    $stmt->execute([$userId, $friendId, $friendId, $userId]);
    $existing = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($existing) {
        if ($existing['status'] === 'accepted') {
            sendErrorResponse('Already friends', 400);
        } else if ($existing['status'] === 'pending') {
            sendErrorResponse('Friend request already sent', 400);
        }
    }
    
    // Send friend request
    $stmt = $db->prepare("
        INSERT INTO friends (user_id, friend_id, status)
        VALUES (?, ?, 'pending')
    ");
    $stmt->execute([$userId, $friendId]);
    
    $data = [
        'friend_id' => $friendId,
        'friend_name' => $friend['name'],
        'status' => 'pending',
    ];
    
    sendSuccessResponse('Friend request sent successfully', $data);
    
} catch (Exception $e) {
    sendErrorResponse($e->getMessage(), 500);
}


