<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/jwt.php';
require_once __DIR__ . '/../../includes/helpers.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    sendErrorResponse('Method not allowed', 405);
}

try {
    // Verify JWT token
    $userId = verifyJWT();
    
    // Get query parameters
    $type = isset($_GET['type']) ? trim($_GET['type']) : 'my'; // my, all, discover
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 20;
    $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
    
    // Validate limit
    if ($limit < 1 || $limit > 100) {
        $limit = 20;
    }
    
    // Get database connection
    $db = getDBConnection();
    
    // Build query based on type
    if ($type === 'my') {
        // Communities user is a member of
        $query = "
            SELECT 
                c.id, c.name, c.description, c.image_url, c.is_private, c.created_by, c.created_at,
                u.name as creator_name,
                (SELECT COUNT(*) FROM community_members WHERE community_id = c.id AND status = 'approved') as members_count,
                (SELECT COUNT(*) FROM community_posts WHERE community_id = c.id) as posts_count,
                cm.role as user_role
            FROM private_communities c
            INNER JOIN users u ON c.created_by = u.id
            INNER JOIN community_members cm ON c.id = cm.community_id
            WHERE cm.user_id = ? AND cm.status = 'approved'
            ORDER BY c.created_at DESC
            LIMIT ? OFFSET ?
        ";
        $params = [$userId, $limit, $offset];
        
        $countQuery = "
            SELECT COUNT(*) as total 
            FROM private_communities c
            INNER JOIN community_members cm ON c.id = cm.community_id
            WHERE cm.user_id = ? AND cm.status = 'approved'
        ";
        $countParams = [$userId];
        
    } else if ($type === 'discover') {
        // Public communities user is not a member of
        $query = "
            SELECT 
                c.id, c.name, c.description, c.image_url, c.is_private, c.created_by, c.created_at,
                u.name as creator_name,
                (SELECT COUNT(*) FROM community_members WHERE community_id = c.id AND status = 'approved') as members_count,
                (SELECT COUNT(*) FROM community_posts WHERE community_id = c.id) as posts_count,
                NULL as user_role
            FROM private_communities c
            INNER JOIN users u ON c.created_by = u.id
            WHERE c.is_private = 0
            AND c.id NOT IN (
                SELECT community_id FROM community_members WHERE user_id = ? AND status = 'approved'
            )
            ORDER BY c.created_at DESC
            LIMIT ? OFFSET ?
        ";
        $params = [$userId, $limit, $offset];
        
        $countQuery = "
            SELECT COUNT(*) as total 
            FROM private_communities c
            WHERE c.is_private = 0
            AND c.id NOT IN (
                SELECT community_id FROM community_members WHERE user_id = ? AND status = 'approved'
            )
        ";
        $countParams = [$userId];
        
    } else {
        // All public communities
        $query = "
            SELECT 
                c.id, c.name, c.description, c.image_url, c.is_private, c.created_by, c.created_at,
                u.name as creator_name,
                (SELECT COUNT(*) FROM community_members WHERE community_id = c.id AND status = 'approved') as members_count,
                (SELECT COUNT(*) FROM community_posts WHERE community_id = c.id) as posts_count,
                (SELECT role FROM community_members WHERE community_id = c.id AND user_id = ? AND status = 'approved') as user_role
            FROM private_communities c
            INNER JOIN users u ON c.created_by = u.id
            WHERE c.is_private = 0
            ORDER BY c.created_at DESC
            LIMIT ? OFFSET ?
        ";
        $params = [$userId, $limit, $offset];
        
        $countQuery = "SELECT COUNT(*) as total FROM private_communities WHERE is_private = 0";
        $countParams = [];
    }
    
    $stmt = $db->prepare($query);
    $stmt->execute($params);
    $communities = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Convert boolean fields
    foreach ($communities as &$community) {
        $community['is_private'] = (bool)$community['is_private'];
        $community['members_count'] = (int)$community['members_count'];
        $community['posts_count'] = (int)$community['posts_count'];
    }
    
    // Get total count
    $countStmt = $db->prepare($countQuery);
    $countStmt->execute($countParams);
    $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    $data = [
        'communities' => $communities,
        'total' => (int)$total,
        'limit' => $limit,
        'offset' => $offset,
    ];
    
    sendSuccessResponse('Communities retrieved successfully', $data);
    
} catch (Exception $e) {
    sendErrorResponse($e->getMessage(), 500);
}


