<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/jwt.php';
require_once __DIR__ . '/../../includes/helpers.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendErrorResponse('Method not allowed', 405);
}

try {
    // Verify JWT token
    $userId = verifyJWT();
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Validate required fields
    if (!isset($input['community_id'])) {
        sendErrorResponse('Community ID is required', 400);
    }
    
    $communityId = (int)$input['community_id'];
    
    // Get database connection
    $db = getDBConnection();
    
    // Check if community exists
    $stmt = $db->prepare("SELECT id, is_private FROM private_communities WHERE id = ?");
    $stmt->execute([$communityId]);
    $community = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$community) {
        sendErrorResponse('Community not found', 404);
    }
    
    // Check if already a member
    $stmt = $db->prepare("SELECT id, status FROM community_members WHERE community_id = ? AND user_id = ?");
    $stmt->execute([$communityId, $userId]);
    $existingMember = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($existingMember) {
        if ($existingMember['status'] === 'approved') {
            sendErrorResponse('Already a member of this community', 400);
        } else if ($existingMember['status'] === 'pending') {
            sendErrorResponse('Join request already pending', 400);
        }
    }
    
    // Determine status based on community privacy
    $status = $community['is_private'] ? 'pending' : 'approved';
    
    // Add member
    $stmt = $db->prepare("
        INSERT INTO community_members (community_id, user_id, role, status)
        VALUES (?, ?, 'member', ?)
    ");
    $stmt->execute([$communityId, $userId, $status]);
    
    $message = $status === 'pending' 
        ? 'Join request sent. Waiting for admin approval.' 
        : 'Successfully joined the community!';
    
    $data = [
        'status' => $status,
        'community_id' => $communityId,
    ];
    
    sendSuccessResponse($message, $data);
    
} catch (Exception $e) {
    sendErrorResponse($e->getMessage(), 500);
}


