<?php
/**
 * Update Task
 * PUT /api/tasks/update-task.php
 */

require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/helpers.php';
require_once __DIR__ . '/../../includes/jwt.php';

header('Content-Type: application/json');
$userId = requireAuth();

try {
    $db = getDBConnection();
    
    // Get PUT data
    $data = json_decode(file_get_contents('php://input'), true);
    
    // Validate required fields
    if (empty($data['id'])) {
        sendResponse(false, 'Task ID is required');
    }
    
    // Check if task exists and belongs to user
    $stmt = $db->prepare("SELECT id FROM tasks WHERE id = ? AND user_id = ?");
    $stmt->execute([$data['id'], $userId]);
    if (!$stmt->fetch()) {
        sendResponse(false, 'Task not found');
    }
    
    // Build update query dynamically
    $updates = [];
    $params = [];
    
    if (isset($data['title'])) {
        if (empty($data['title'])) {
            sendResponse(false, 'Title cannot be empty');
        }
        $updates[] = "title = ?";
        $params[] = $data['title'];
    }
    
    if (isset($data['description'])) {
        $updates[] = "description = ?";
        $params[] = $data['description'];
    }
    
    if (isset($data['category'])) {
        $validCategories = ['seeding', 'harvesting', 'bd500', 'bd501', 'cpp', 'watering', 'other'];
        if (!in_array($data['category'], $validCategories)) {
            sendResponse(false, 'Invalid category');
        }
        $updates[] = "category = ?";
        $params[] = $data['category'];
    }

    if (isset($data['status'])) {
        $validStatuses = ['pending', 'completed', 'postponed', 'cancelled'];
        if (!in_array($data['status'], $validStatuses)) {
            sendResponse(false, 'Invalid status');
        }
        $updates[] = "status = ?";
        $params[] = $data['status'];
    }
    
    if (isset($data['priority'])) {
        $validPriorities = ['low', 'medium', 'high'];
        if (!in_array($data['priority'], $validPriorities)) {
            sendResponse(false, 'Invalid priority');
        }
        $updates[] = "priority = ?";
        $params[] = $data['priority'];
    }
    
    if (isset($data['due_date'])) {
        if (!empty($data['due_date'])) {
            $dueDate = DateTime::createFromFormat('Y-m-d', $data['due_date']);
            if (!$dueDate) {
                sendResponse(false, 'Invalid due date format. Use YYYY-MM-DD');
            }
        }
        $updates[] = "due_date = ?";
        $params[] = $data['due_date'] ?: null;
    }
    
    if (isset($data['reminder_date'])) {
        if (!empty($data['reminder_date'])) {
            $reminderDate = DateTime::createFromFormat('Y-m-d H:i:s', $data['reminder_date']);
            if (!$reminderDate) {
                sendResponse(false, 'Invalid reminder date format. Use YYYY-MM-DD HH:MM:SS');
            }
        }
        $updates[] = "reminder_date = ?";
        $params[] = $data['reminder_date'] ?: null;
    }
    
    if (isset($data['is_recurring'])) {
        $updates[] = "is_recurring = ?";
        $params[] = (bool)$data['is_recurring'] ? 1 : 0;
    }
    
    if (isset($data['recurrence_pattern'])) {
        $updates[] = "recurrence_pattern = ?";
        $params[] = !empty($data['recurrence_pattern']) ? json_encode($data['recurrence_pattern']) : null;
    }
    
    if (isset($data['crop_id'])) {
        if (!empty($data['crop_id'])) {
            $stmt = $db->prepare("SELECT id FROM crops WHERE id = ? AND user_id = ?");
            $stmt->execute([$data['crop_id'], $userId]);
            if (!$stmt->fetch()) {
                sendResponse(false, 'Invalid crop ID');
            }
        }
        $updates[] = "crop_id = ?";
        $params[] = $data['crop_id'] ?: null;
    }
    
    if (isset($data['farm_section_id'])) {
        if (!empty($data['farm_section_id'])) {
            $stmt = $db->prepare("SELECT id FROM farm_sections WHERE id = ? AND user_id = ?");
            $stmt->execute([$data['farm_section_id'], $userId]);
            if (!$stmt->fetch()) {
                sendResponse(false, 'Invalid farm section ID');
            }
        }
        $updates[] = "farm_section_id = ?";
        $params[] = $data['farm_section_id'] ?: null;
    }
    
    if (empty($updates)) {
        sendResponse(false, 'No fields to update');
    }
    
    // Add updated_at
    $updates[] = "updated_at = NOW()";
    
    // Update task
    $query = "UPDATE tasks SET " . implode(', ', $updates) . " WHERE id = ? AND user_id = ?";
    $params[] = $data['id'];
    $params[] = $userId;
    
    $stmt = $db->prepare($query);
    $stmt->execute($params);
    
    // Get the updated task
    $stmt = $db->prepare("
        SELECT
            t.id,
            t.title,
            t.description,
            t.category,
            t.status,
            t.priority,
            t.due_date,
            t.reminder_date,
            t.is_recurring,
            t.recurrence_pattern,
            t.created_at,
            t.updated_at,
            c.id as crop_id,
            c.crop_name,
            fs.id as farm_section_id,
            fs.section_name
        FROM tasks t
        LEFT JOIN crops c ON t.crop_id = c.id
        LEFT JOIN farm_sections fs ON t.farm_section_id = fs.id
        WHERE t.id = ?
    ");
    
    $stmt->execute([$data['id']]);
    $task = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Convert boolean fields
    $task['is_recurring'] = (bool)$task['is_recurring'];
    
    // Parse recurrence pattern JSON
    if ($task['recurrence_pattern']) {
        $task['recurrence_pattern'] = json_decode($task['recurrence_pattern'], true);
    }
    
    sendResponse(true, 'Task updated successfully', ['task' => $task]);
    
} catch (PDOException $e) {
    error_log('Database error: ' . $e->getMessage());
    sendResponse(false, 'Failed to update task');
} catch (Exception $e) {
    error_log('Error: ' . $e->getMessage());
    sendResponse(false, 'An error occurred');
}


