<?php
/**
 * Populate BioDynamic Calendar with Sample Data
 * This script generates calendar data for the current year
 */

require_once '../config/database.php';

echo "BioDynamic Calendar - Data Population\n";
echo "=====================================\n\n";

try {
    $db = getDBConnection();
    
    // Lunar phases
    $lunarPhases = ['New Moon', 'Waxing Crescent', 'First Quarter', 'Waxing Gibbous', 
                    'Full Moon', 'Waning Gibbous', 'Last Quarter', 'Waning Crescent'];
    
    // Zodiac signs
    $zodiacSigns = ['Aries', 'Taurus', 'Gemini', 'Cancer', 'Leo', 'Virgo', 
                    'Libra', 'Scorpio', 'Sagittarius', 'Capricorn', 'Aquarius', 'Pisces'];
    
    // Elements
    $elements = ['Fire', 'Earth', 'Air', 'Water'];
    
    // Element to zodiac mapping
    $elementZodiac = [
        'Fire' => ['Aries', 'Leo', 'Sagittarius'],
        'Earth' => ['Taurus', 'Virgo', 'Capricorn'],
        'Air' => ['Gemini', 'Libra', 'Aquarius'],
        'Water' => ['Cancer', 'Scorpio', 'Pisces']
    ];
    
    // Recommendations by element
    $recommendations = [
        'Fire' => ['Good for fruit crops', 'Favorable for seed planting', 'Ideal for harvesting fruits'],
        'Earth' => ['Excellent for root vegetables', 'Good for transplanting', 'Favorable for soil preparation'],
        'Air' => ['Best for flower planting', 'Good for pruning', 'Ideal for harvesting flowers'],
        'Water' => ['Perfect for leafy vegetables', 'Good for watering', 'Favorable for composting']
    ];
    
    $year = date('Y');
    $startDate = "$year-01-01";
    $endDate = "$year-12-31";
    
    echo "Generating calendar data for year $year...\n";
    
    // Clear existing data for the year
    $stmt = $db->prepare("DELETE FROM biodynamic_calendar WHERE YEAR(`date`) = ?");
    $stmt->execute([$year]);
    
    $stmt = $db->prepare("
        INSERT INTO biodynamic_calendar (
            date, lunar_phase, zodiac_sign, element,
            is_favorable_planting, is_favorable_harvesting,
            is_favorable_pruning, is_favorable_fertilizing,
            recommendations, notes
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    $currentDate = new DateTime($startDate);
    $endDateTime = new DateTime($endDate);
    $dayCount = 0;
    
    while ($currentDate <= $endDateTime) {
        $dateStr = $currentDate->format('Y-m-d');
        $dayOfYear = (int)$currentDate->format('z');
        
        // Calculate lunar phase (simplified - cycles every 29.5 days)
        $lunarPhaseIndex = floor(($dayOfYear % 29.5) / 3.69);
        $lunarPhase = $lunarPhases[$lunarPhaseIndex];
        
        // Calculate zodiac sign (simplified - changes every ~30 days)
        $zodiacIndex = floor($dayOfYear / 30.4) % 12;
        $zodiacSign = $zodiacSigns[$zodiacIndex];
        
        // Determine element based on zodiac
        $element = '';
        foreach ($elementZodiac as $elem => $signs) {
            if (in_array($zodiacSign, $signs)) {
                $element = $elem;
                break;
            }
        }
        
        // Determine favorable activities based on lunar phase and element
        $isFavorablePlanting = in_array($lunarPhase, ['New Moon', 'Waxing Crescent', 'First Quarter']) ? 1 : 0;
        $isFavorableHarvesting = in_array($lunarPhase, ['Full Moon', 'Waning Gibbous']) ? 1 : 0;
        $isFavorablePruning = in_array($lunarPhase, ['Waning Gibbous', 'Last Quarter', 'Waning Crescent']) ? 1 : 0;
        $isFavorableFertilizing = in_array($lunarPhase, ['Waxing Gibbous', 'Full Moon']) ? 1 : 0;
        
        // Get recommendations for the element
        $recs = $recommendations[$element] ?? [];
        $recsJson = json_encode($recs);
        
        // Generate notes
        $notes = "Lunar phase: $lunarPhase. Element: $element. Zodiac: $zodiacSign.";
        
        $stmt->execute([
            $dateStr,
            $lunarPhase,
            $zodiacSign,
            $element,
            $isFavorablePlanting,
            $isFavorableHarvesting,
            $isFavorablePruning,
            $isFavorableFertilizing,
            $recsJson,
            $notes
        ]);
        
        $dayCount++;
        $currentDate->modify('+1 day');
    }
    
    echo "\n";
    echo "Calendar data populated successfully!\n";
    echo "Total days: $dayCount\n";
    echo "Year: $year\n";
    echo "\n";
    
} catch (PDOException $e) {
    echo "Database error: " . $e->getMessage() . "\n";
    exit(1);
} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    exit(1);
}

